/*-------------------------------------------------------------------------
 *
 * pg_attrmask.h
 *	  definition of the "attribute masks" system catalog (pg_attrmask)
 *
 *
 * Portions Copyright (c) 1996-2022, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 * src/include/catalog/pg_attrmask.h
 *
 * NOTES
 *	  The Catalog.pm module reads this file and derives schema
 *	  information.
 *
 *-------------------------------------------------------------------------
 */
#ifndef PG_ATTRMASK_H
#define PG_ATTRMASK_H

#include "catalog/genbki.h"
#include "catalog/objectaddress.h"
#include "catalog/pg_attrmask_d.h"
#include "utils/relcache.h"

/* ----------------
 *		pg_attrmask definition.  cpp turns this into
 *		typedef struct FormData_pg_attrmask
 * ----------------
 */
CATALOG(pg_attrmask,6122,AttrMaskRelationId)
{
	Oid			oid;			/* oid */

	Oid			amrelid BKI_LOOKUP(pg_class);	/* OID of table containing
												 * attribute */
	int16		amnum;			/* attnum of attribute */

#ifdef CATALOG_VARLEN			/* variable-length fields start here */
	pg_node_tree ambin BKI_FORCE_NOT_NULL;	/* nodeToString representation of
											 * the masking expression */
#endif
} FormData_pg_attrmask;

/* ----------------
 *		Form_pg_attrmask corresponds to a pointer to a tuple with
 *		the format of pg_attrmask relation.
 * ----------------
 */
typedef FormData_pg_attrmask *Form_pg_attrmask;

DECLARE_TOAST(pg_attrmask, 6123, 6124);

DECLARE_UNIQUE_INDEX(pg_attrmask_amrelid_amnum_index, 6125, AttrMaskIndexId, pg_attrmask, btree(amrelid oid_ops, amnum int2_ops));
DECLARE_UNIQUE_INDEX_PKEY(pg_attrmask_oid_index, 6126, AttrMaskOidIndexId, pg_attrmask, btree(oid oid_ops));

DECLARE_FOREIGN_KEY((amrelid, amnum), pg_attribute, (attrelid, attnum));

/*
 * Since the C source for the caches (syscache_info.h and syscache_ids.h) is
 * generated automatically and since the IDs are ordered alphabetically, we
 * add the 'ZZZ' prefix to make sure they end up at the end of the list. This
 * is to avoid renumbering of the core caches. If the cache IDs were
 * renumbered, then extensions compiled for "vanilla PG" would not work with
 * PGEE.
 */
MAKE_SYSCACHE(ZZZ_ATTMASK, pg_attrmask_amrelid_amnum_index, 64);
MAKE_SYSCACHE(ZZZ_ATTMASKOID, pg_attrmask_oid_index, 64);

extern void StoreAttrMask(Relation rel, AttrNumber attnum, Node *expr,
						  bool is_raw);
extern void RemoveAttrMask(Oid relid, AttrNumber attnum,
						   DropBehavior behavior, bool complain,
						   bool internal);
extern void RemoveAttrMaskById(Oid attrmaskId);
extern ObjectAddress GetAttrMaskedColumnAddress(Oid attrmaskoid);
#endif							/* PG_ATTRMASK_H */
